using System;
using System.Collections;
using System.Data;
using System.Text;
using System.Text.RegularExpressions;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.VAL;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// This class is used by UC16 to manage orders.
	/// </summary>
	public class Orders 
	{
		#region variables
		private const string LAB_TEST_TABLE_NAME = "LabTest";

		// represents the currently selected order
		private BOL.PatientOrder _patientOrder;

		// These two tables store all of the orders
		DataTable _dtComponentOrders;
		DataTable _dtTestOrders;

		// Regular expressions for search
		private Regex _lastfiveEx;
		private Regex _ssnEx;
		private Regex _nameEx;

		// Contains all of the filteres orders from the search
		ArrayList _filteredOrders;

		//for report
		private bool _searchByPatient;
		private bool _searchBySpecimen;
		private bool _searchByDate;
		private bool _searchByOrder;

		/// <summary>
		/// OrderSelectedEventHandler
		/// </summary>
		public delegate void OrderSelectedEventHandler(object sender, SelectOrderEventArgs e);
		
		/// <summary>
		/// Event raised when orders are selected
		/// </summary>
		public event OrderSelectedEventHandler OrderSelectedEvent;

		/// <summary>
		/// OrderSelected
		/// </summary>
		/// <param name="orderStatus"></param>
		protected virtual void OrderSelected(Common.AcceptOrderStatus orderStatus)
		{
			SelectOrderEventArgs oEA = new SelectOrderEventArgs(orderStatus);
			if (OrderSelectedEvent != null)
				OrderSelectedEvent(this, oEA);
		}

		/// <summary>
		/// This event holds messages related to the issue of units.
		/// </summary>
		public class SelectOrderEventArgs : System.EventArgs 
		{

			private Common.AcceptOrderStatus _acceptOrderStatus;

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/11/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7534"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7535"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// SelectOrderEventArgs
			/// </summary>
			/// <param name="acceptOrderStatus"></param>
			public SelectOrderEventArgs(Common.AcceptOrderStatus acceptOrderStatus) 
			{
				_acceptOrderStatus = acceptOrderStatus;
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/11/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7536"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7537"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// AcceptOrderStatus
			/// </summary>
			public Common.AcceptOrderStatus AcceptOrderStatus
			{
				get
				{
					return _acceptOrderStatus;
				}
			}
		}
			
		
		#endregion


		#region constructors

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5978"> 
		///		<ExpectedInput>None, but working in a valid division</ExpectedInput>
		///		<ExpectedOutput>Orders object with multiple rows in DataTables</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5979"> 
		///		<ExpectedInput>None, but working in an invalid division</ExpectedInput>
		///		<ExpectedOutput>Orders object with no rows in DataTables</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Blank Constructor
		/// </summary>
		public Orders()
		{
			// Fill DataTables
			_dtComponentOrders = BOL.OrderedComponent.GetComponentOrders();
			_dtTestOrders = BOL.OrderedTest.GetTestOrders();

			_filteredOrders = new ArrayList();

			_lastfiveEx = Common.RegularExpressions.PatientInitialAndLastFourSSN();
			_ssnEx = Common.RegularExpressions.PatientSsn();
			_nameEx = Common.RegularExpressions.PatientName();

			// Get initial List
			ApplyFilter(string.Empty, false, DateTime.MinValue, DateTime.MinValue, string.Empty, string.Empty);
		}

		#endregion

		
		#region properties

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5976"> 
		///		<ExpectedInput>A valid component or test order</ExpectedInput>
		///		<ExpectedOutput>SelectedOrder set to the order</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5977"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This is the order that is currently selected in the UI.  This may
		/// be a test or component order
		/// </summary>
		public BOL.PatientOrder SelectedOrder
		{
			get
			{
				return _patientOrder;
			}
			set
			{
				_patientOrder = value;
				// raise event
				OrderSelected(CheckOrder());
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5974"> 
		///		<ExpectedInput>Valid Order and working in a valid division</ExpectedInput>
		///		<ExpectedOutput>Count is > 0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5975"> 
		///		<ExpectedInput>Valid Order, but working in an invalid division</ExpectedInput>
		///		<ExpectedOutput>Count is 0</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Contains all of the orders for display.
		/// </summary>
		public ArrayList DisplayOrders
		{
			get
			{
				return _filteredOrders;
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5972"> 
		///		<ExpectedInput>Valid Order and working in a valid division</ExpectedInput>
		///		<ExpectedOutput>Count is > 0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5973"> 
		///		<ExpectedInput>Valid Order, but working in an invalid division</ExpectedInput>
		///		<ExpectedOutput>Count is 0</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DataTable containing all component orders (pre-filter)
		/// </summary>
		public DataTable AllComponentOrders
		{
			get
			{
				return _dtComponentOrders;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5970"> 
		///		<ExpectedInput>Valid Order and working in a valid division</ExpectedInput>
		///		<ExpectedOutput>Count is > 0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5971"> 
		///		<ExpectedInput>Valid Order, but working in an invalid division</ExpectedInput>
		///		<ExpectedOutput>Count is 0</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DataTable containing all test orders (pre-filter)
		/// </summary>
		public DataTable AllTestOrders
		{
			get
			{
				return _dtTestOrders;
			}
		}

		#endregion


		#region methods


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5968"> 
		///		<ExpectedInput>Valid search parameter</ExpectedInput>
		///		<ExpectedOutput>Count > 0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5969"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Count = 0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Filters orders based on search parameters entered by the patient
		/// </summary>
		/// 
		/// <param name="specimenUid"></param>
		/// <param name="searchByDate"></param>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="patient"></param>
		/// <param name="labOrderNumber"></param>
		/// <returns></returns>
		public void FilterOrdersForDisplay(string specimenUid, 
			bool searchByDate,
			DateTime startDate,
			DateTime endDate, 
			string patient, 
			string labOrderNumber)
		{
			ApplyFilter(specimenUid.Trim(), searchByDate, startDate, endDate, patient.Trim(), labOrderNumber.Trim());
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5966"> 
		///		<ExpectedInput>Valid CPRS Number</ExpectedInput>
		///		<ExpectedOutput>Row count stays the same when reject set to false or decreased
		///		by one when reject set to true.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5967"> 
		///		<ExpectedInput>null CPRSNumber as argument</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Removes and oRder
		/// </summary>
		/// <param name="cprsOrderNumber"></param>
		/// <param name="reject"></param>
		public void Remove(string cprsOrderNumber, bool reject)
		{
			if (cprsOrderNumber == null || cprsOrderNumber == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("cprsOrderNumber").ResString);
			}

			foreach(DataRow dr in _filteredOrders)
			{
				if(dr[TABLE.OrderedComponent.CprsOrderNumber].ToString() == cprsOrderNumber)
				{
					if (reject)
					{
						_filteredOrders.Remove(dr);
					}
					else
					{
						dr[TABLE.OrderedComponent.OrderStatusCode] = Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Accepted);
					}
					break;
				}
			}
		}

		/// <summary>
		/// Returns the status of the order letting the UI know if it's OK to accept, additional
		/// information is needed, or that the order cannot be accepted.
		/// BR_16.06
		/// </summary>
		/// <returns></returns>
		private Common.AcceptOrderStatus CheckOrder()
		{
			// No order selected
			if (_patientOrder == null)
			{
				return Common.AcceptOrderStatus.None;
			}

			if( !VistALink.EnsureAvailability() )
				return Common.AcceptOrderStatus.None;;

			CheckLab();

			bool isComponent = false;
			string labOrderNumber = string.Empty;
			string labTestId = string.Empty;

			// Get accession information
			if (_patientOrder is BOL.OrderedComponent) 
			{
				isComponent = true;
				labOrderNumber = ( (BOL.OrderedComponent) _patientOrder ).LabOrderNumber.Trim();
				labTestId = ( (BOL.OrderedComponent) _patientOrder ).LabTestId.Trim();
				((BOL.OrderedComponent) _patientOrder).SpecimenAccessionNumber = BOL.OrderedComponent.GetAccessionByLabOrderNumber(labOrderNumber.Trim(), labTestId.Trim());
			}
			else
			{
				labOrderNumber = ( (BOL.OrderedTest) _patientOrder ).LabOrderNumber.Trim();
				labTestId = ( (BOL.OrderedTest) _patientOrder ).LabTestId.Trim();
				((BOL.OrderedTest) _patientOrder).SpecimenAccessionNumber = BOL.OrderedComponent.GetAccessionByLabOrderNumber(labOrderNumber.Trim(), labTestId.Trim());
			}			

			
			
			// Order not in Pending status
			if( (isComponent && ( (BOL.OrderedComponent) _patientOrder).OrderStatusCode != Common.OrderStatus.Pending) ||
				( !isComponent && ((BOL.OrderedTest) _patientOrder).OrderStatusCode != Common.OrderStatus.Pending))
			{
				return Common.AcceptOrderStatus.AlreadyAcceptedOrRejected;
			}

			// Order hasn't been accessioned
			else if ( (isComponent && ((BOL.OrderedComponent) _patientOrder).SpecimenAccessionNumber == string.Empty) ||
				( !isComponent && ((BOL.OrderedTest) _patientOrder).SpecimenAccessionNumber == string.Empty))
			{
				return Common.AcceptOrderStatus.NotAccessioned;
			}

			// Now, do all of the specimen processing
			// If a test, a new specimen must always be identified:
			if (!isComponent)
			{
				if ( ((BOL.OrderedTest) _patientOrder).Specimen.SpecimenUid == string.Empty )
				{
					return Common.AcceptOrderStatus.RequiresSpecimen;
				}
				else
				{
					return ProcessTestSpecimen(labOrderNumber, labTestId);
				}
			}
			// BR_16.06
			else if ( ( (BOL.OrderedComponent) _patientOrder).SpecimenRequiredIndicator )
			{
				if (( (BOL.OrderedComponent) _patientOrder).SpecimenIdentified) // CR3114
				{
					return ProcessComponentSpecimen();
				}
				else
				{
					return Common.AcceptOrderStatus.RequiresSpecimen;
				}
			}

			else
			{
				return Common.AcceptOrderStatus.ReadyForAcceptance;
			}	
		}

		///<Developers>
		///	<Developer>COE User</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/30/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8134"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8135"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BR_16.14
		/// </summary>
		/// <returns></returns>
		public Common.AcceptOrderStatus ProcessComponentSpecimen()
		{
			BOL.Specimen specimen = ( (BOL.OrderedComponent) _patientOrder).GetSpecimen(); // CR3114

			specimen.SetInitialBrokenRules("FrmMaintainSpecimen", "ExistingSpecimen");
			// Trigger broken rules
			specimen.SpecimenUid = specimen.SpecimenUid;
			specimen.IsAssociatedWithPendingOrder = true;
			specimen.UnacceptableSpecimenIndicator = specimen.UnacceptableSpecimenIndicator;

			// New specimen
			if (specimen.IsNew)
			{
				return Common.AcceptOrderStatus.SpecimenNotInVBECS;
			}
			// Specimen must be assigned to the current patient
			if (specimen.Patient.VistaPatientId != _patientOrder.Patient.VistaPatientId)
			{
				return Common.AcceptOrderStatus.WrongSpecimen;
			}
			// Specimen must be maintainable (TAS)
			if (!specimen.MaintainableSpecimenIndicator)
			{
				return Common.AcceptOrderStatus.SpecimenNotMaintainable;
			}
			// // BR_16.14 Rules broken
			if (specimen.BrokenRulesCount > 0)
			{
				return Common.AcceptOrderStatus.SpecimenHasBrokenRules;
			}

			return Common.AcceptOrderStatus.ReadyForAcceptance;
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8244"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8245"> 
		///		<ExpectedInput>Invalid lab information</ExpectedInput>
		///		<ExpectedOutput>NoLabDataForSpecimen</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a status for the order based on specimen information
		/// BR_16.14, BR_16.19
		/// </summary>
		/// <param name="labOrderNumber"></param>
		/// <param name="labTestId"></param>
		/// <returns></returns>
		public Common.AcceptOrderStatus ProcessTestSpecimen(string labOrderNumber, string labTestId)
		{
			BOL.Specimen specimen = ( (BOL.OrderedTest) _patientOrder).Specimen;

			specimen.SetInitialBrokenRules("FrmMaintainSpecimen", "ExistingSpecimen");

			specimen.SpecimenUid = specimen.SpecimenUid;
			specimen.IsAssociatedWithPendingOrder = true;
			specimen.UnacceptableSpecimenIndicator = specimen.UnacceptableSpecimenIndicator;


			DataSet dsAccession = DAL.VAL.LabOrder.GetLabAccessionByLabOrderNumber(labOrderNumber);
			DataRow dtRow = null;
			bool found = false;
			//Address CR 1749
			foreach( DataTable dt in dsAccession.Tables )
			{
				if( found )
					break;

				if( dt.TableName == LAB_TEST_TABLE_NAME )
				{
					foreach (DataRow dr in dt.Rows)
					{

						if( dr.Table.Columns.Contains(TABLE.OrderedComponent.LabOrderNumber) && !dr.IsNull(TABLE.OrderedComponent.LabOrderNumber) 
							&& dr.Table.Columns.Contains(TABLE.OrderedComponent.LabTestId) && !dr.IsNull(TABLE.OrderedComponent.LabTestId)
							&& dr.Table.Columns.Contains(TABLE.OrderedComponent.SpecimenAccessionNumber) && !dr.IsNull(TABLE.OrderedComponent.SpecimenAccessionNumber)
							&& labOrderNumber == dr[TABLE.OrderedComponent.LabOrderNumber].ToString().Trim()
							&& labTestId == dr[TABLE.OrderedComponent.LabTestId].ToString().Trim() )
						{
						#if DEBUG
						Console.WriteLine("Use this UID: " + dr[TABLE.PatientSpecimen.SpecimenUid]);
						#endif

							// BR_16.19
							if ( ((BOL.OrderedTest) _patientOrder).Specimen.SpecimenUid != dr[TABLE.PatientSpecimen.SpecimenUid].ToString().Trim())
							{
								return Common.AcceptOrderStatus.SpecimenUidInvalid;
							}
							dtRow = dr;
							found = true;
							break;
			
						}
						else
						{

						}
						
					}
				}
			}
			if (!found)
			{
				if (specimen.IsNew)
				{
					return Common.AcceptOrderStatus.NoLabDataForSpecimen;
				}
				else
				{
					return Common.AcceptOrderStatus.SpecimenMismatch;
				}
			}

            if(dtRow == null || dtRow[Common.DatabaseConstants.VistALinkColumnNames.CollectionDateTime] == null)
                throw new ApplicationException("Row is null or CollectionDateTime is null");

			// for patient specimen record
			string collectionDateTime = dtRow[Common.DatabaseConstants.VistALinkColumnNames.CollectionDateTime].ToString();
			
			// IF NON MAINTAINABLE TEST (NOT TAS)
			if ( ( (BOL.OrderedTest) _patientOrder).OrderableTestId != (int) Common.OrderableTest.TAS)
			{
				// CR 2817
				((BOL.OrderedTest) _patientOrder).Specimen.SpecimenCollectionDate = Common.HL7DateFormat.ConvertHL7DateTime(collectionDateTime);
				//
				return Common.AcceptOrderStatus.ReadyForAcceptance;
			}
			else
			{
				if ( !specimen.IsNew && specimen.BrokenRulesCount==0)
				{
					return Common.AcceptOrderStatus.ReadyForAcceptance;
				}
				else if (specimen.IsNew || specimen.BrokenRulesCount > 0)
				{
					((BOL.OrderedTest) _patientOrder).Specimen.SpecimenCollectionDate = Common.HL7DateFormat.ConvertHL7DateTime(collectionDateTime);
					//
					return Common.AcceptOrderStatus.SpecimenRequiresUpdate;
				}
			}

			return Common.AcceptOrderStatus.ReadyForAcceptance;
		}


		/// <summary>
		/// All orders must come across with a lab test id and lab order number!
		/// This checks to make sure data isn't old or bad.
		/// </summary>
		private void CheckLab()
		{
			if (_patientOrder is BOL.OrderedComponent)
			{
				BOL.OrderedComponent oC = (BOL.OrderedComponent) _patientOrder;

				if ( (oC.LabTestId == null || oC.LabTestId.Trim() == string.Empty) ||
					(oC.LabOrderNumber == null || oC.LabOrderNumber.Trim() == string.Empty))
				{
					throw new BOL.BusinessObjectException("Lab information is incomplete");
				}
			}
			else
			{
				BOL.OrderedTest oT = (BOL.OrderedTest) _patientOrder;
				
				if ( (oT.LabTestId == null || oT.LabTestId.Trim() == string.Empty) ||
					(oT.LabOrderNumber == null || oT.LabOrderNumber.Trim() == string.Empty) )
				{
					throw new BOL.BusinessObjectException("Lab information is incomplete");
				}
			}
		}


		/// <summary>
		/// BR_88.02 - This method filters orders depending on search criteria.  It then builds an 
		/// ArrayList containing these orders.  If no search criteria are entered, all rows are 
		/// returned.
		/// </summary>
		private ArrayList ApplyFilter(string specimenUid, 
			bool searchByDate,
			DateTime startDate,
			DateTime endDate, 
			string patient, 
			string labOrderNumber)
		{
			_filteredOrders.Clear();

			_searchByPatient = false;
			_searchBySpecimen = false;
			_searchByDate = false;
			_searchByOrder = false;

			StringBuilder filterStringComponents = new StringBuilder();
			StringBuilder filterStringTests = new StringBuilder();
			// specimen
			if (specimenUid.Length > 0)
			{			
				_searchBySpecimen = true;

				string patientID = null;
				DataSet dsLabOrder = DAL.VAL.LabOrder.GetLabOrderNumberBySpecimenUID(specimenUid);
				if (dsLabOrder.Tables.Count == 3)
				{
					patientID = (string) dsLabOrder.Tables[0].Rows[0][TABLE.Patient.VistaPatientId];
				}
				if (patientID != null)
				{
					string statement = "(" + TABLE.Patient.VistaPatientId + " = '" + patientID + "'" + ") ";
					filterStringComponents.Append(statement.ToString());
					filterStringTests.Append(statement.ToString());
				}
					// no matches
				else
				{
					return _filteredOrders;
				}				
			}

			// date
			if (searchByDate)
			{
				_searchByDate = true;
				
				if (filterStringComponents.Length > 0) 
				{
					filterStringComponents.Append("AND ");
					filterStringTests.Append("AND ");
				}
				filterStringComponents.Append("(" + TABLE.OrderedComponent.RequiredDatetime +  " >= '" + startDate.Date + "' AND " +  
					TABLE.OrderedComponent.RequiredDatetime + " <= '" + endDate.Date.AddHours(23).AddMinutes(59).AddSeconds(59) +"') ");
				filterStringTests.Append("(" + TABLE.OrderedTest.RequiredResultsDate + " >= '" + startDate.Date + "' AND " + 
					TABLE.OrderedTest.RequiredResultsDate + " <= '" + endDate.Date.AddHours(23).AddMinutes(59).AddSeconds(59) +"') ");
			}

			// patient
			if (patient.Length > 0)
			{			
				_searchByPatient = true;

				if (filterStringComponents.Length > 0) 
				{
					filterStringComponents.Append("AND ");
					filterStringTests.Append("AND ");
				}
				
				patient = patient.ToUpper();
				if (_lastfiveEx.IsMatch(patient))
				{
					filterStringComponents.Append(TABLE.Patient.PatientLastName + " LIKE '" + patient.Substring(0,1) + 
						"%' AND " + TABLE.Patient.PatientSsn + " LIKE '%" + patient.Substring(1,4) + "' ");
					filterStringTests.Append(TABLE.Patient.PatientLastName + " LIKE '" + patient.Substring(0,1) + 
						"%' AND " + TABLE.Patient.PatientSsn + " LIKE '%" + patient.Substring(1,4) + "' ");
				}
				else if(this._nameEx.IsMatch(patient))
				{
					filterStringComponents.Append(TABLE.Patient.PatientLastName + " LIKE '" + patient + "%' ");
					filterStringTests.Append(TABLE.Patient.PatientLastName + " LIKE '" + patient + "%' ");
				}
				else if (this._ssnEx.IsMatch(patient))
				{
					filterStringComponents.Append(TABLE.Patient.PatientSsn + " LIKE '%" + patient + "%' ");
					filterStringTests.Append(TABLE.Patient.PatientSsn + " LIKE '%" + patient + "%' ");
				}
			}

			// order ID
			if (labOrderNumber.Length > 0)
			{ 
				_searchByOrder = true;
				if (filterStringComponents.Length > 0) 
				{
					filterStringComponents.Append("AND ");
					filterStringTests.Append("AND ");
				}
				filterStringComponents.Append(TABLE.OrderedComponent.LabOrderNumber + " LIKE '%" + labOrderNumber + "%' ");
				filterStringTests.Append(TABLE.OrderedTest.LabOrderNumber + " LIKE '%" + labOrderNumber + "%' ");
			}

			// BR_88.04
			if (filterStringComponents.Length > 0) 
			{
				filterStringComponents.Append("AND ");
				filterStringTests.Append("AND ");
			}
			filterStringComponents.Append("(" + TABLE.OrderedComponent.OrderStatusCode + " = 'A' OR " +
				TABLE.OrderedComponent.OrderStatusCode + " = 'P')");
			filterStringTests.Append("(" + TABLE.OrderedTest.OrderStatusCode + " = 'A' OR " +
				TABLE.OrderedTest.OrderStatusCode + " = 'P')");

			DataRow[] drComponents = _dtComponentOrders.Select(filterStringComponents.ToString(),Common.DatabaseConstants.ArtificialColumnNames.UrgencySortColumn+","+Common.DatabaseConstants.ArtificialColumnNames.RequestSortColumn+","+TABLE.OrderedComponent.RequiredDatetime);
			DataRow[] drTests = _dtTestOrders.Select(filterStringTests.ToString(),Common.DatabaseConstants.ArtificialColumnNames.UrgencySortColumn+","+Common.DatabaseConstants.ArtificialColumnNames.RequestSortColumn+","+TABLE.OrderedTest.RequiredResultsDate);
	
			DataTable dtComponentOrdersFiltered = _dtComponentOrders.Clone();
			DataTable dtTestOrdersFiltered = _dtTestOrders.Clone();

			foreach(DataRow dr in drComponents)
			{
				System.Data.DataRow dtRow = dtComponentOrdersFiltered.NewRow();
				dtRow.ItemArray = dr.ItemArray;
				dtComponentOrdersFiltered.Rows.Add(dtRow);
			}
			foreach(DataRow dr in drTests)
			{
				System.Data.DataRow dtRow = dtTestOrdersFiltered.NewRow();
				dtRow.ItemArray = dr.ItemArray;
				dtTestOrdersFiltered.Rows.Add(dtRow);
			}

			foreach (DataRow dtRow in dtTestOrdersFiltered.Rows)
			{
				_filteredOrders.Add(dtRow);
			}
			foreach (DataRow dtRow in dtComponentOrdersFiltered.Rows)
			{
				_filteredOrders.Add(dtRow);
			}

			//_filteredOrders = SortOrders(_filteredOrders);

			return _filteredOrders;
		}


		#region printing

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5964"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5965"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Report printing
		/// </summary>
		public void PrintReport(System.Drawing.Printing.PrinterSettings settings)
		{
			BOL.OrderedComponent orderedComponent = null;
			BOL.OrderedTest orderedTest = null;
		
			bool isOrderedComponent = false; 

			DataSet dsPrint = new DataSet();

			DataTable dtPrint = new DataTable();
			dtPrint.Columns.Add(TABLE.PatientTreatment.PatientLocation);
			dtPrint.Columns.Add(TABLE.ExceptionPatientTest.PatientName);
			dtPrint.Columns.Add(TABLE.ExceptionSpecimen.PatientId);
			dtPrint.Columns.Add(TABLE.ExceptionExpiredTask.Quantity);
			dtPrint.Columns.Add(ARTIFICIAL.OrderingMD);
			dtPrint.Columns.Add(ARTIFICIAL.OrderedDate, typeof(DateTime));
			dtPrint.Columns.Add(ARTIFICIAL.WantedDate, typeof(DateTime));

			dtPrint.Columns.Add(ARTIFICIAL.OrderUrgency);
			dtPrint.Columns.Add(ARTIFICIAL.SpecimenStatus);
			dtPrint.Columns.Add(ARTIFICIAL.Procedure);
			// foo
			dtPrint.Columns.Add("NoInappropriateReportText");
			dtPrint.Columns.Add(TABLE.OrderedTest.LabOrderNumber);
			dtPrint.Columns.Add(TABLE.PatientOrder.OrderGroupNumber);

        
			for(int i=0; i< _filteredOrders.Count; i++)
			{
				BOL.PatientOrder patientOrder = new BOL.PatientOrder((DataRow) _filteredOrders[i]);
				if (((DataRow) _filteredOrders[i]).Table.Columns.Contains(TABLE.ComponentClass.ComponentClassName)) 
				{
					isOrderedComponent = true;
					orderedComponent = new BOL.OrderedComponent((DataRow) _filteredOrders[i]);//(BOL.OrderedComponent) SelectedPatientOrder;
					if (orderedComponent.OrderStatusCode != Common.OrderStatus.Pending) continue;
				}
				else
				{
					isOrderedComponent = false;
					orderedTest = new BOL.OrderedTest((DataRow) _filteredOrders[i]);//(BOL.OrderedTest) SelectedPatientOrder;
					if (orderedTest.OrderStatusCode != Common.OrderStatus.Pending) continue;
				}

				DataRow drPrint = dtPrint.NewRow();
                if (!((DataRow)_filteredOrders[i]).IsNull(VbecsTables.PatientTreatment.PatientLocation))
                    drPrint[TABLE.PatientTreatment.PatientLocation] = (string)((DataRow)_filteredOrders[i])[VbecsTables.PatientTreatment.PatientLocation];
                else
                    drPrint[TABLE.PatientTreatment.PatientLocation] = "";

				drPrint[TABLE.ExceptionPatientTest.PatientName] = patientOrder.Patient.DisplayName;
				drPrint[TABLE.ExceptionSpecimen.PatientId] = patientOrder.Patient.DisplayPatientID;//SSN.Substring(0,3) + "-" + SelectedPatientOrder.Patient.SSN.Substring(3,2)+ "-" + SelectedPatientOrder.Patient.SSN.Substring(5,4);
				//				
				drPrint[ARTIFICIAL.OrderingMD] = patientOrder.OrderingProviderLastName.Trim() + ", " + patientOrder.OrderingProviderFirstName.Trim();
				drPrint[ARTIFICIAL.OrderedDate] = patientOrder.OrderReceivedDateTime;
				drPrint[TABLE.PatientOrder.OrderGroupNumber] = patientOrder.OrderGroupNumber;
			
				if (isOrderedComponent)
				{
                    //CR 3554
                    if (orderedComponent == null || orderedComponent.ComponentOrdered == null)
                        throw new ApplicationException("orderedComponent is null");

					if (orderedComponent.ComponentOrdered.IsSpecimenRequired == true)
					{
						drPrint[ARTIFICIAL.SpecimenStatus] = "Required";
					}
					else
					{
						drPrint[ARTIFICIAL.SpecimenStatus] = "Not required";
					}
					drPrint[TABLE.ExceptionExpiredTask.Quantity] = orderedComponent.Quantity;
					if (orderedComponent.RequiredDateTime > DateTime.MinValue)
					{
						drPrint[ARTIFICIAL.WantedDate] = orderedComponent.RequiredDateTime;
					}
					drPrint[ARTIFICIAL.OrderUrgency] = orderedComponent.OrderUrgencyText.Trim();
					drPrint[ARTIFICIAL.Procedure] = orderedComponent.ComponentOrdered.BloodComponentName;
					drPrint[TABLE.OrderedTest.LabOrderNumber] = orderedComponent.LabOrderNumber.Trim();
				}
				else
				{
                    //CR 3554
                    if (orderedTest == null)
                        throw new ApplicationException("orderedTest is null");

					drPrint[ARTIFICIAL.SpecimenStatus] = "Required";
					if (orderedTest.RequiredResultsDate > DateTime.MinValue)
					{
						drPrint[ARTIFICIAL.WantedDate] = orderedTest.RequiredResultsDate;
					}
					drPrint[ARTIFICIAL.OrderUrgency] = orderedTest.OrderUrgencyText;
					drPrint[ARTIFICIAL.Procedure] = orderedTest.OrderableTest;
					drPrint[TABLE.ExceptionExpiredTask.Quantity] = null;
					drPrint[TABLE.OrderedTest.LabOrderNumber] = orderedTest.LabOrderNumber.Trim();
				}
				dtPrint.Rows.Add(drPrint);
			}

			StringBuilder criteria = new StringBuilder();
		
			if (_searchBySpecimen == true) criteria.Append("Specimen");
			if (_searchByDate == true)
			{
				if (criteria.Length == 0) criteria.Append("Date");
				else criteria.Append(", \nDate");
			}
			if (_searchByPatient == true)
			{
				if (criteria.Length == 0) criteria.Append("Patient");
				else criteria.Append(", \nPatient");
			}
			if (_searchByOrder == true)
			{
				if (criteria.Length == 0) criteria.Append("Order");
				else criteria.Append(", \nOrder");
			}
		
			dsPrint.Tables.Add(dtPrint);

#if DEBUG
			// Only used to generate initial report template
			//dtPrint.WriteXml("e:\\temp\\PendingOrderListDataSet.xml", System.Data.XmlWriteMode.WriteSchema); 
#endif

			DataSet dsHeader =  BOL.VbecsReport.GetReportHeader(Common.LogonUser.LogonUserName);
			DataSet dsFooter = BOL.VbecsReport.GetReportFooter(criteria.ToString());

            ReportDataSource reportSourceMain = new ReportDataSource();
            reportSourceMain.Name = "PendingOrderListDataSet";
            reportSourceMain.Value = dtPrint;

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }

            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsFooter.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom("Reports.dll");
            Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.PendingOrderList.PendingOrderList.rdlc");

            LocalReport report = new LocalReport();

            report.DataSources.Add(reportSourceHeader);
            report.DataSources.Add(reportSourceFooter);
            report.DataSources.Add(reportSourceMain);
            report.LoadReportDefinition(stream);

		    using (var reportPrinter = new VbecsReportPrinter(settings, true, report))
		    {
		        reportPrinter.Print();
		    }
		}

		#endregion

		#endregion
	}
}
